/*:
 * @target MZ
 * @plugindesc Picture buttons that run common events in PARALLEL during messages. v1.3a null-safe consume flag.
 * @author HS
 *
 * @param cooldownMs
 * @text 連打ガード(ミリ秒)
 * @type number
 * @default 400
 *
 * @param dedupe
 * @text 同一CEの連打を抑止
 * @type boolean
 * @default true
 *
 * @param acceptRightClick
 * @text 右クリックも受付
 * @type boolean
 * @default true
 *
 * @param blockDuringChoice
 * @text 選択肢表示中は無効
 * @type boolean
 * @default true
 *
 * @param consumeFrames
 * @text メッセ送り抑止フレーム
 * @type number
 * @default 3
 *
 * @command bind
 * @text ピクチャにCEをバインド
 * @arg pictureId
 * @type number
 * @default 1
 * @arg commonEventId
 * @type common_event
 * @default 1
 *
 * @command unbind
 * @text バインド解除
 * @arg pictureId
 * @type number
 * @default 1
 */
(() => {
  const PN = "HS_ButtonPictureRealtime";
  const P  = PluginManager.parameters(PN);
  const COOLDOWN     = Number(P.cooldownMs || 0);
  const DEDUPE       = String(P.dedupe) === "true";
  const RIGHTCLICK   = String(P.acceptRightClick) === "true";
  const BLOCK_CHOICE = String(P.blockDuringChoice) === "true";
  const CONSUME_FR   = Math.max(0, Number(P.consumeFrames || 0));

  // ─────────────────────────────────────────
  // $gameTemp が未初期化でも動く「クリック抑止」ストレージ
  const CF = {
    _fallback: 0,
    get() { return (window.$gameTemp ? ($gameTemp._hsui_consumeFrames | 0) : (this._fallback | 0)); },
    set(v){ if (window.$gameTemp) $gameTemp._hsui_consumeFrames = v; else this._fallback = v; },
    add(v){ if (v>0) this.set(Math.max(this.get(), v)); },
    dec(){ const n = this.get(); if (n>0) this.set(n-1); }
  };
  // ゲームオブジェクト生成時にフォールバック値を移行
  const _DM_cgo = DataManager.createGameObjects;
  DataManager.createGameObjects = function(){
    _DM_cgo.call(this);
    if (CF._fallback > 0) $gameTemp._hsui_consumeFrames = CF._fallback;
  };

  // ─────────────────────────────────────────
  // 並列実行インタプリタ
  const HSUI = {
    queue: [],
    interpreter: new Game_Interpreter(),
    enqueue(id){ if(!(DEDUPE && this.queue.includes(id))) this.queue.push(id); },
    update(){
      if (this.interpreter.isRunning()) this.interpreter.update();
      else if (this.queue.length>0){
        const id = this.queue.shift();
        const ce = $dataCommonEvents[id];
        if (ce && ce.list) this.interpreter.setup(ce.list, 0);
      }
    },
    consume(fr=CONSUME_FR){
      CF.add(fr);
      // クリックの残り火を即消去
      if (TouchInput._triggered) TouchInput._triggered = false;
      if (TouchInput._released)  TouchInput._released  = false;
      if (TouchInput._cancelled) TouchInput._cancelled = false;
    }
  };

  // プラグインコマンド
  PluginManager.registerCommand(PN, "bind", args => {
    const pic = $gameScreen.picture(Number(args.pictureId));
    if (pic){ pic._hs_rt_ceId = Number(args.commonEventId); pic._hs_rt_last = 0; }
  });
  PluginManager.registerCommand(PN, "unbind", args => {
    const pic = $gameScreen.picture(Number(args.pictureId));
    if (pic){ delete pic._hs_rt_ceId; delete pic._hs_rt_last; }
  });

  // ─────────────────────────────────────────
  // クリック先取り：シーン更新の最初で判定→消費
  function preUpdateHit(scene){
    if (!scene || !scene._spriteset) return;
    if (BLOCK_CHOICE && $gameMessage && $gameMessage.isBusy() && ($gameMessage._choices?.length>0)) return;

    const trig = TouchInput.isTriggered();
    const rclk = RIGHTCLICK && TouchInput.isCancelled();
    if (!trig && !rclk) return;

    const cont = scene._spriteset._pictureContainer;
    if (!cont) return;
    const list = cont.children ?? [];
    const px = TouchInput.x, py = TouchInput.y;

    for (let i=list.length-1; i>=0; i--){
      const sp  = list[i];
      const pic = sp?.picture?.() ?? null;
      if (!sp || !pic || !sp.visible || !sp.worldVisible) continue;
      const ceId = pic._hs_rt_ceId;
      if (!ceId) continue;

      // ワールド座標ヒット
      const b = sp.getBounds?.();
      const hit = b && px >= b.x && py >= b.y && px < b.x + b.width && py < b.y + b.height;
      if (!hit) continue;

      // クールダウン
      const now = Date.now();
      if (COOLDOWN>0 && pic._hs_rt_last && (now - pic._hs_rt_last) < COOLDOWN) return;
      pic._hs_rt_last = now;

      HSUI.consume();
      HSUI.enqueue(ceId);
      SoundManager.playOk();
      break;
    }
  }

  // Scene_Map / Scene_Battle の update 先頭で先取り
  const _SM_update = Scene_Map.prototype.update;
  Scene_Map.prototype.update = function(){ preUpdateHit(this); _SM_update.call(this); HSUI.update(); };
  const _SB_update = Scene_Battle.prototype.update;
  Scene_Battle.prototype.update = function(){ preUpdateHit(this); _SB_update.call(this); HSUI.update(); };

  // 念のため Spriteset 側でカウントダウンだけ進める
  const _SBa_update = Spriteset_Base.prototype.update;
  Spriteset_Base.prototype.update = function(){ _SBa_update.call(this); CF.dec(); };

  // メッセージ系の入力を抑止（null安全版）
  const _TI_isTrig   = TouchInput.isTriggered;
  const _TI_isRep    = TouchInput.isRepeated;
  const _TI_isPressed= TouchInput.isPressed;
  TouchInput.isTriggered = function(){ if (CF.get()>0) return false; return _TI_isTrig.call(this); };
  TouchInput.isRepeated  = function(){ if (CF.get()>0) return false; return _TI_isRep.call(this); };
  TouchInput.isPressed   = function(){ if (CF.get()>0) return false; return _TI_isPressed.call(this); };

  const _IN_isTrig = Input.isTriggered;
  const _IN_isRep  = Input.isRepeated;
  Input.isTriggered = function(symbol){
    if (CF.get()>0 && (symbol==='ok' || symbol==='cancel')) return false;
    return _IN_isTrig.call(this, symbol);
  };
  Input.isRepeated = function(symbol){
    if (CF.get()>0 && (symbol==='ok' || symbol==='cancel')) return false;
    return _IN_isRep.call(this, symbol);
  };

  const _WM_processInput = Window_Message.prototype.processInput;
  Window_Message.prototype.processInput = function(){
    if (CF.get()>0) return false;
    return _WM_processInput.call(this);
  };
})();
